<?php
// --- 1. إعدادات قاعدة البيانات وجلب الجلسات ---
$host = 'localhost';
$dbname = 'shafpknj_arabstrg';
$username = 'shafpknj_ahmed';
$password = '2719243214kK';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("فشل الاتصال بقاعدة البيانات: " . $e->getMessage());
}

// معالجة طلب الحفظ (AJAX)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_to_db') {
    header('Content-Type: application/json');
    try {
        $sessionId = filter_input(INPUT_POST, 'session_id', FILTER_VALIDATE_INT);
        $jsonData = $_POST['data'] ?? '';

        if (!$sessionId || empty($jsonData)) {
            throw new Exception("بيانات غير صالحة. تأكد من اختيار جلسة وتوفر البيانات.");
        }

        $data = json_decode($jsonData, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("تنسيق JSON غير صالح.");
        }

        $pdo->beginTransaction();

        // حذف البيانات القديمة لهذه الجلسة لتجنب التكرار (اختياري، حسب رغبتك في التحديث أو الاستبدال)
        $stmtDelete = $pdo->prepare("DELETE FROM session_country_stats WHERE session_id = ?");
        $stmtDelete->execute([$sessionId]);

        $stmtInsert = $pdo->prepare("
            INSERT INTO session_country_stats (
                session_id, country_tag, country_name,
                factories_civilian, factories_military, factories_naval,
                gdp_total, gdp_per_capita, overall_productivity,
                defence_breakdown_airforce, defence_breakdown_industry,
                defence_breakdown_industry_gdp_modifier,
                defence_breakdown_land, defence_breakdown_navy,
                defence_gain, debt, imp_refugees_taken_k
            ) VALUES (
                ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?
            )
        ");

        $insertedCount = 0;
        foreach ($data as $row) {
            // التأكد من وجود tag الدولة لأنه أساسي
            if (empty($row['code'])) continue;

            $stmtInsert->execute([
                $sessionId,
                $row['code'],
                $row['name'] ?? null,
                (int)($row['factories']['civilian'] ?? 0),
                (int)($row['factories']['military'] ?? 0),
                (int)($row['factories']['naval'] ?? 0),
                (float)($row['gdp_total'] ?? 0),
                (float)($row['gdp_per_capita'] ?? 0),
                (float)($row['overall_productivity'] ?? 0),
                (float)($row['defence_breakdown_airforce'] ?? 0),
                (float)($row['defence_breakdown_industry'] ?? 0),
                ($row['defence_breakdown_industry_gdp_modifier'] ?? '0'),
                (float)($row['defence_breakdown_land'] ?? 0),
                (float)($row['defence_breakdown_navy'] ?? 0),
                (float)($row['defence_gain'] ?? 0),
                (float)($row['debt'] ?? 0),
                (float)($row['imp_refugees_taken_k'] ?? 0)
            ]);
            $insertedCount++;
        }

        $pdo->commit();
        echo json_encode(['success' => true, 'message' => "تم حفظ بيانات $insertedCount دولة بنجاح للجلسة رقم $sessionId."]);
    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => "خطأ: " . $e->getMessage()]);
    }
    exit;
}

// جلب قائمة الجلسات للعرض في القائمة المنسدلة
$sessions = [];
try {
    $stmt = $pdo->query("SELECT session_id, session_name, session_date FROM sessions ORDER BY session_date DESC");
    $sessions = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // يمكن تجاهل الخطأ هنا أو عرضه بطريقة مناسبة
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl" class="dark">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>محلل بيانات HOI4 المتقدم</title>
    <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap" rel="stylesheet">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    <style>
        :root {
            --primary: #4CAF50; --primary-hover: #45a049;
            --bg: #121212; --card: #1e1e1e; --text: #ffffff; --border: #333;
            --highlight: #2d2d2d; --edit: #3a3a3a;
            --th-bg: #1a1a1a;
        }
        * { margin: 0; padding: 0; box-sizing: border-box; font-family: 'Cairo', sans-serif; }
        body { background: var(--bg); color: var(--text); padding: 20px; line-height: 1.4; }
        .container { max-width: 100%; margin: 0 auto; padding: 0 10px; }
        h1 { color: var(--primary); text-align: center; margin-bottom: 20px; font-size: 1.8rem; }
        .card { background: var(--card); padding: 15px; border-radius: 10px; margin-bottom: 20px; box-shadow: 0 4px 6px rgba(0,0,0,0.2); }
        .btn-group { display: flex; gap: 10px; justify-content: center; flex-wrap: wrap; margin-top: 15px; align-items: center; }
        button { padding: 8px 16px; border: none; border-radius: 6px; color: #fff; cursor: pointer; transition: 0.2s; font-weight: 600; font-size: 0.9rem; }
        .btn-blue { background: #007bff; } .btn-blue:hover { background: #0056b3; }
        .btn-green { background: #28a745; } .btn-green:hover { background: #218838; }
        .btn-red { background: #dc3545; } .btn-red:hover { background: #c82333; }
        .btn-yellow { background: #ffc107; color: #000; } .btn-yellow:hover { background: #e0a800; }
        input, select { background: var(--card); border: 1px solid var(--border); color: var(--text); padding: 8px; border-radius: 6px; outline: none; }
        input:focus, select:focus { border-color: var(--primary); }
        .controls { margin-bottom: 15px; }
        .admin-panel { border: 1px solid var(--primary); padding: 15px; margin-top: 20px; position: relative; }
        .admin-panel::before { content: 'لوحة تحكم المشرف'; position: absolute; top: -12px; right: 20px; background: var(--bg); padding: 0 10px; color: var(--primary); font-size: 0.9em; }

        /* === إصلاحات الجدول والتثبيت === */
        .table-wrap { overflow: auto; max-height: 75vh; border-radius: 8px; border: 1px solid var(--border); position: relative; }
        table { width: 100%; border-collapse: separate; border-spacing: 0; table-layout: auto; }
        th, td { padding: 10px 6px; text-align: center; border-bottom: 1px solid var(--border); background: var(--card); font-size: 0.9rem; white-space: normal; word-wrap: break-word; vertical-align: middle; }
        th { background: var(--th-bg) !important; color: var(--primary); position: sticky; top: 0; z-index: 100; border-top: none; border-bottom: 2px solid var(--primary); box-shadow: 0 2px 4px rgba(0,0,0,0.3); }
        td.editable:hover { background: var(--edit); cursor: pointer; outline: 1px solid var(--primary); }
        td input { text-align: center; background: var(--highlight); border: none; padding: 4px; width: 100%; font-size: 0.9rem; color: var(--text); }
        .hidden { display: none !important; }
        .loader { width: 40px; height: 40px; border: 4px solid var(--card); border-top-color: var(--primary); border-radius: 50%; animation: spin 1s linear infinite; margin: 20px auto; }
        @keyframes spin { to { transform: rotate(360deg); } }
        .suffix { font-size: 0.8em; opacity: 0.7; margin-right: 2px; }
    </style>
</head>
<body>
<div class="container">
    <h1>محلل بيانات HOI4 المتقدم</h1>

    <div class="card">
        <input type="file" id="fileInput" accept=".txt,.hoi4,.json,.xlsx,.xls" style="width: 100%; margin-bottom: 10px;">
        <div class="btn-group">
            <button class="btn-yellow" onclick="document.getElementById('fileInput').click()">📂 فتح ملف</button>
            <button class="btn-blue" onclick="app.exportToExcel()">📥 تصدير إكسل</button>
            <button class="btn-green" onclick="app.saveJson()">💾 حفظ JSON محلياً</button>
        </div>
    </div>

    <div class="card admin-panel">
        <div class="btn-group" style="justify-content: space-between;">
            <select id="sessionIdSelect" style="flex-grow: 1; max-width: 400px;">
                <option value="">-- اختر الجلسة للحفظ في قاعدة البيانات --</option>
                <?php foreach ($sessions as $session): ?>
                    <option value="<?= $session['session_id'] ?>">
                        <?= htmlspecialchars($session['session_name'] . ' (' . $session['session_date'] . ')') ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <button id="saveToDbBtn" class="btn-red" onclick="app.saveToDatabase()">☁️ حفظ البيانات في قاعدة البيانات</button>
        </div>
    </div>

    <div class="loader hidden" id="loader"></div>

    <div class="controls card">
        <input type="text" id="filter" placeholder="🔍 بحث عن دولة..." style="width: 100%;">
    </div>

    <div class="table-wrap card">
        <table id="mainTable"></table>
    </div>
</div>

<script>
// نفس تكوين الأعمدة والدول السابق
const APP_CONFIG = {
    cols: {
        "name": { label: "الدولة", type: "text" },
        "factories.civilian": { label: "مصانع مدنية 😃", type: "int" },
        "factories.military": { label: "مصانع عسكرية 💥", type: "int" },
        "factories.naval": { label: "أحواض سفن 🚢", type: "int" },
        "gdp_total": { label: "GDP 💲", type: "money" },
        "gdp_per_capita": { label: "نصيب الفرد 📊", type: "k_suffix" },
        "overall_productivity": { label: "الإنتاجية ⚙️", type: "float" },
        "defence_breakdown_airforce": { label: "تكاليف جوية ✈️", type: "money_suffix" },
        "defence_breakdown_industry": { label: "تكاليف صناعة 🏭", type: "money_suffix" },
        "defence_breakdown_industry_gdp_modifier": { label: "إنفاق عسكري % 📈", type: "float" },
        "defence_breakdown_land": { label: "تكاليف برية 🚚", type: "money_suffix" },
        "defence_breakdown_navy": { label: "تكاليف بحرية ⚓", type: "money_suffix" },
        "defence_gain": { label: "دفاع كلي 💣", type: "money_suffix" },
        "debt": { label: "ديون 💸", type: "money_suffix" },
        "imp_refugees_taken_k": { label: "لاجئين (ألف)", type: "k_suffix" }
    },
    countries: {
        "AFG": "Afghanistan", "ALB": "Albania", "ALG": "Algeria", "ARG": "Argentina", "ARM": "Armenia", "AST": "Australia", "AUS": "Austria", "AZE": "Azerbaijan", "BEL": "Belgium", "BFA": "Burkina Faso", "BLR": "Belarus", "BOL": "Bolivia", "BRA": "Brazil", "BRM": "Myanmar", "BUL": "Bulgaria", "CAN": "Canada", "CHI": "China", "CHL": "Chile", "COL": "Colombia", "COS": "Costa Rica", "CRO": "Croatia", "CUB": "Cuba", "CZE": "Czech Republic", "DEN": "Denmark", "DOM": "Dominican Republic", "DRC": "Congo DR", "ECU": "Ecuador", "EGY": "Egypt", "ELS": "El Salvador", "ENG": "United Kingdom", "EST": "Estonia", "ETH": "Ethiopia", "FIN": "Finland", "FRA": "France", "GEO": "Georgia", "GER": "Germany", "GRE": "Greece", "GUA": "Guatemala", "HAI": "Haiti", "HOL": "Netherlands", "HON": "Honduras", "HUN": "Hungary", "IND": "Indonesia", "IRE": "Ireland", "IRQ": "Iraq", "ISR": "Israel", "ITA": "Italy", "JAP": "Japan", "KAZ": "Kazakhstan", "KOR": "South Korea", "LAT": "Latvia", "LBA": "Libya", "LIB": "Liberia", "LIT": "Lithuania", "LUX": "Luxembourg", "MEX": "Mexico", "MON": "Mongolia", "MOR": "Morocco", "NEP": "Nepal", "NIC": "Nicaragua", "NIG": "Nigeria", "NOR": "Norway", "NZL": "New Zealand", "OMA": "Oman", "PAK": "Pakistan", "PAN": "Panama", "PAR": "Paraguay", "PER": "Iran", "PHI": "Philippines", "POL": "Poland", "POR": "Portugal", "PRU": "Peru", "RAJ": "India", "ROM": "Romania", "SAF": "South Africa", "SAU": "Saudi Arabia", "SER": "Yugoslavia", "SIA": "Thailand", "SIN": "Singapore", "SLO": "Slovakia", "SOV": "Russia", "SPR": "Spain", "SWE": "Sweden", "SWI": "Switzerland", "SYR": "Syria", "TUR": "Turkey", "UAE": "UAE", "UKR": "Ukraine", "URU": "Uruguay", "USA": "United States", "UZB": "Uzbekistan", "VEN": "Venezuela", "YEM": "Yemen", "YUG": "Yugoslavia"
    }
};

const app = {
    data: new Map(),
    sortDir: {},

    init() {
        this.cacheDOM();
        this.bindEvents();
        this.renderTableHead();
    },
    cacheDOM() {
        this.els = {
            file: document.getElementById('fileInput'),
            loader: document.getElementById('loader'),
            filter: document.getElementById('filter'),
            table: document.getElementById('mainTable'),
            sessionSelect: document.getElementById('sessionIdSelect'),
            saveDbBtn: document.getElementById('saveToDbBtn')
        };
    },
    bindEvents() {
        this.els.file.addEventListener('change', (e) => this.handleFile(e.target.files[0]));
        this.els.filter.addEventListener('input', (e) => this.filterTable(e.target.value));
    },

    // ================== CORE ==================
    async handleFile(file) {
        if (!file) return;
        this.toggleLoader(true);
        try {
            if (file.name.match(/\.(txt|hoi4)$/i)) await this.processHoi4(file);
            else if (file.name.endsWith('.json')) await this.processJson(file);
            else if (file.name.match(/\.xls[x]?$/i)) await this.processExcel(file);
            this.renderTableBody();
            alert("✅ تم تحميل البيانات بنجاح");
        } catch (e) {
            alert("❌ خطأ: " + e.message);
            console.error(e);
        }
        this.toggleLoader(false);
        this.els.file.value = '';
    },
    async processHoi4(file) {
        const text = await file.text();
        this.data.clear();
        const playerTags = [...text.matchAll(/(\w{3})\s*=\s*\{\s*cosmetic_tag=\1/g)].map(m => m[1]);
        if (playerTags.length === 0) {
             const pcBlock = text.match(/player_countries\s*=\s*\{([\s\S]*?)\n\}/)?.[1] || "";
             [...pcBlock.matchAll(/(\w{3})\s*=\s*\{/g)].forEach(m => playerTags.push(m[1]));
        }
        [...new Set(playerTags)].forEach(tag => {
            if (APP_CONFIG.countries[tag] || playerTags.length < 50) {
                 this.data.set(tag, this.extractCountryData(text, tag));
            }
        });
    },
    extractCountryData(text, tag) {
        const variations = [tag, `${tag}_AUTH`, `${tag}_democratic`, `${tag}_communism`, "UAR"];
        let block = null;
        for (const t of variations) {
            const tagPos = text.indexOf(`cosmetic_tag="${t}"`);
            if (tagPos === -1) continue;
            const varStart = text.lastIndexOf("variables={", tagPos);
            if (varStart !== -1) {
                block = this.getBalancedBlock(text, varStart);
                if (block && block.includes("gdp_total")) break;
            }
        }
        const data = { code: tag, name: APP_CONFIG.countries[tag] || tag }; // تمت إضافة code هنا
        if (!block) return this.getEmptyData(tag);
        for (const [key, col] of Object.entries(APP_CONFIG.cols)) {
            if (key === 'name') continue;
            const prop = key.includes('.') ? key.split('.')[1] : key;
            const saveKey = key.includes('factories') ? `effective_${prop}_factories` : key;
            const match = block.match(new RegExp(`${saveKey}\\s*=\\s*([\\d\\.]+)`));
            this.setValue(data, key, match ? match[1] : '0');
        }
        return data;
    },
    getBalancedBlock(text, start) {
        let depth = 0;
        for (let i = start; i < text.length; i++) {
            if (text[i] === '{') depth++;
            else if (text[i] === '}') {
                depth--;
                if (depth === 0) return text.substring(start, i + 1);
            }
        }
        return null;
    },
    processJson(file) {
        return new Promise((resolve, reject) => {
            const r = new FileReader();
            r.onload = e => {
                try {
                    const json = JSON.parse(e.target.result);
                    this.data.clear();
                    json.forEach(item => this.data.set(item.code || item.name, item));
                    resolve();
                } catch (err) { reject(err); }
            };
            r.readAsText(file);
        });
    },
    processExcel(file) {
         return new Promise((resolve, reject) => {
            const r = new FileReader();
            r.onload = e => {
                try {
                    const wb = XLSX.read(new Uint8Array(e.target.result), {type: 'array'});
                    const json = XLSX.utils.sheet_to_json(wb.Sheets[wb.SheetNames[0]]);
                    this.data.clear();
                    json.forEach(row => {
                        const code = Object.keys(APP_CONFIG.countries).find(k => APP_CONFIG.countries[k] === row[APP_CONFIG.cols.name.label]) || row.Tag || `UNK_${Math.random()}`;
                        const data = { code: code, name: APP_CONFIG.countries[code] || code };
                        Object.entries(APP_CONFIG.cols).forEach(([key, col]) => {
                            if (key !== 'name') this.setValue(data, key, row[col.label] || 0);
                        });
                        this.data.set(code, data);
                    });
                    resolve();
                } catch (err) { reject(err); }
            };
            r.readAsArrayBuffer(file);
        });
    },

    // ================== UI ==================
    renderTableHead() {
        let ths = '';
        Object.entries(APP_CONFIG.cols).forEach(([key, col]) => {
            ths += `<th onclick="app.sortTable('${key}')">${col.label}</th>`;
        });
        this.els.table.innerHTML = `<thead><tr>${ths}</tr></thead><tbody id="tBody"></tbody>`;
    },
    renderTableBody() {
        const tbody = document.getElementById('tBody');
        tbody.innerHTML = '';
        this.data.forEach((row, code) => {
            const tr = document.createElement('tr');
            Object.entries(APP_CONFIG.cols).forEach(([key, col]) => {
                const td = document.createElement('td');
                const val = this.getValue(row, key);
                td.innerHTML = this.formatValue(val, col.type);
                if (key !== 'name') {
                    td.className = 'editable';
                    td.onclick = () => this.editCell(td, code, key, col.type);
                }
                tr.appendChild(td);
            });
            tbody.appendChild(tr);
        });
    },
    editCell(td, code, key, type) {
        if (td.querySelector('input')) return;
        const oldHtml = td.innerHTML;
        const rawValue = this.getValue(this.data.get(code), key);
        td.innerHTML = `<input type="text" value="${rawValue}">`;
        const input = td.querySelector('input');
        input.focus();
        input.onblur = input.onkeydown = (e) => {
            if (e.type === 'blur' || e.key === 'Enter') {
                let newVal = input.value.replace(/[^\d\.-]/g, '');
                if (newVal === '') newVal = '0';
                this.setValue(this.data.get(code), key, newVal);
                td.innerHTML = this.formatValue(newVal, type);
            } else if (e.key === 'Escape') td.innerHTML = oldHtml;
        };
    },

    // ================== UTILS ==================
    getValue(obj, path) { return path.split('.').reduce((o, i) => o?.[i], obj); },
    setValue(obj, path, val) {
        const parts = path.split('.');
        let target = obj;
        for (let i = 0; i < parts.length - 1; i++) {
            target[parts[i]] = target[parts[i]] || {};
            target = target[parts[i]];
        }
        target[parts[parts.length - 1]] = val;
    },
    formatValue(val, type) {
        const n = parseFloat(val);
        if (isNaN(n)) return val;
        switch (type) {
            case 'money': return Math.round(n).toLocaleString() + '<span class="suffix">b</span>';
            case 'money_suffix': return n.toFixed(2) + '<span class="suffix">b</span>';
            case 'k_suffix': return Math.round(n).toLocaleString() + '<span class="suffix">K</span>';
            case 'float': return n.toFixed(1);
            case 'int': return Math.round(n).toLocaleString();
            default: return val;
        }
    },
    getEmptyData(tag) {
        const data = { code: tag, name: APP_CONFIG.countries[tag] || tag, factories: {} };
        Object.keys(APP_CONFIG.cols).forEach(k => k !== 'name' && this.setValue(data, k, 'N/A'));
        return data;
    },
    sortTable(key) {
        this.sortDir[key] = !this.sortDir[key];
        const sorted = [...this.data.entries()].sort((a, b) => {
            const vA = this.getValue(a[1], key), vB = this.getValue(b[1], key);
            const valA = isNaN(parseFloat(vA)) ? vA : parseFloat(vA);
            const valB = isNaN(parseFloat(vB)) ? vB : parseFloat(vB);
            return (valA > valB ? 1 : -1) * (this.sortDir[key] ? 1 : -1);
        });
        this.data = new Map(sorted);
        this.renderTableBody();
    },
    filterTable(txt) {
        const term = txt.toLowerCase();
        [...document.querySelectorAll('#tBody tr')].forEach(tr => {
            tr.style.display = tr.cells[0].textContent.toLowerCase().includes(term) ? '' : 'none';
        });
    },
    toggleLoader(show) { this.els.loader.classList.toggle('hidden', !show); },
    
    // ================== EXPORT & SAVE ==================
    exportToExcel() {
        if (this.data.size === 0) return alert("لا توجد بيانات للتصدير");
        const rows = [...this.data.values()].map(d => {
            const row = {};
            Object.entries(APP_CONFIG.cols).forEach(([k, col]) => row[col.label] = this.getValue(d, k));
            return row;
        });
        const wb = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(wb, XLSX.utils.json_to_sheet(rows), "HOI4 Data");
        XLSX.writeFile(wb, "HOI4_Stats.xlsx");
    },
    saveJson() {
        if (this.data.size === 0) return alert("لا توجد بيانات للحفظ");
        const data = [...this.data.values()]; // استخدام values مباشرة لأننا أضفنا 'code' إلى الكائن
        const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        const a = document.createElement('a');
        a.href = URL.createObjectURL(blob);
        a.download = 'hoi4_data.json';
        a.click();
    },
    async saveToDatabase() {
        if (this.data.size === 0) return alert("⚠️ لا توجد بيانات حالياً للحفظ.");
        
        const sessionId = this.els.sessionSelect.value;
        if (!sessionId) return alert("⚠️ الرجاء اختيار الجلسة أولاً من القائمة.");

        if (!confirm(`هل أنت متأكد من رغبتك في حفظ بيانات ${this.data.size} دولة للجلسة المحددة؟\nسيتم استبدال أي بيانات سابقة لهذه الجلسة.`)) return;

        this.toggleLoader(true);
        this.els.saveDbBtn.disabled = true;
        this.els.saveDbBtn.textContent = "جاري الحفظ...";

        try {
            const dataToSend = [...this.data.values()];
            const formData = new FormData();
            formData.append('action', 'save_to_db');
            formData.append('session_id', sessionId);
            formData.append('data', JSON.stringify(dataToSend));

            const response = await fetch(window.location.href, { // إرسال لنفس الصفحة
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                throw new Error(`HTTP Error: ${response.status}`);
            }

            const result = await response.json();
            if (result.success) {
                alert("✅ " + result.message);
            } else {
                throw new Error(result.message || "فشلت عملية الحفظ لسبب غير معروف.");
            }

        } catch (error) {
            console.error("Error saving to DB:", error);
            alert("❌ حدث خطأ أثناء الحفظ:\n" + error.message);
        } finally {
            this.toggleLoader(false);
            this.els.saveDbBtn.disabled = false;
            this.els.saveDbBtn.innerHTML = "☁️ حفظ البيانات في قاعدة البيانات";
        }
    }
};

document.addEventListener('DOMContentLoaded', () => app.init());
</script>
</body>
</html>